/**
 *
 * \file        joinstate.cpp
 * \brief       memorizes the join states and outputs them when room availab.e
 * \author      Pete McCormick / Larry Salant
 * \date        1/15/2008
 *
 */

#include <stdlib.h>
#include "nutypedefs.h"
#include "cresnet_slave.h"
#include "Cresnet.h"
#include "errors.h"
#include "mytji.h"
#include "tji.h"
#include "console.h"
#include "memorymanager.h"
#include "cresnet_host.hpp"
#include "DMCardJoins.h"
#include "dm_netmapper.hpp"

#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT

#include "CDMSwitchCard.h"
#include "DMController.h"
#include "CDmNetDiscover.h"

#endif


////////////////////////////////////////////////////////////////////////////////
// internal command for queueing online / offline status
#define UNENCAP_COMMAND 3
// Encapsulated commands for online /offline status
const UINT8 EncapOnlinePacket[] = {CNET_ID_CTRLSYS,0x07,CRESNET_ENCAPSULATED,DM_NET_ID,4,INTERNAL,INTERNAL_SUBCMD_DIGITAL,0,DIGI_OFF};
const UINT8 EncapOfflinePacket[] = {CNET_ID_CTRLSYS,0x07,CRESNET_ENCAPSULATED,DM_NET_ID,4,INTERNAL,INTERNAL_SUBCMD_DIGITAL,0,DIGI_ON};



/**
* \author    Larry Salant
* \brief     queue the online status of the downstream device to the controller
* \date      2/21/09
* \param     Stream - which stream to send it to
* \param     online - true if device is present
* \return    void
* \retval    none
**/
void SendOnlineStatusToController(UINT8 bStream, bool online)
{
    if ( online )
        GetState(bStream)->CresnetCommandMemorize((UINT8 *)&EncapOnlinePacket[UNENCAP_COMMAND], NET_DEST_CRESNET, 0);
    else
        GetState(bStream)->CresnetCommandMemorize((UINT8 *)&EncapOfflinePacket[UNENCAP_COMMAND], NET_DEST_CRESNET, 0);
}

////////////////////////////////////////////////////////////////////////////////
/**
 * \author    Larry Salant
 * \brief     constructor for class to memorize join states - used for reporting join changes and
 *            for TJI
 * \date      5/8/2008
 * \param     m_Stream : which stream this join is for
 * \param     NumDigitals : max number of digital joins (16 bit)
 * \param     NumAnalogs : max number of analog joins (16 bit)
 * \param     NumSerials : max number of serial joins
 * \param     NumCommands : max number of command responses
 * \return    void
 * \retval    none
 * \note      Jan 13 2009, R.H: Changed digital/analog joins size to 16 bit.
**/
CresnetJoinState::CresnetJoinState(UINT8 bStreamNum, UINT16 NumDigitals, UINT16 NumAnalogs, UINT16 NumSerials, UINT8 NumCommands, UINT8 bPackJoinLimit) /* = 0 */ :
m_iCommandRequestAckTimer(DM_LIBRARY_DEBOUNCE_DISABLED),
m_bCommandRequestAckSource(NET_DEST_ALL)
{
    // remember the stream number
    m_Stream = bStreamNum;
    m_NumDigitals = NumDigitals;      // max number of digitals
    m_NumAnalogs = NumAnalogs;
    m_NumSerials = NumSerials;
    m_NumCommands = NumCommands;
    m_bPackJoinLimit = bPackJoinLimit;

    //Initialize the counts
    m_bDigitalJoinCount = 0;
    m_bAnalogJoinCount = 0;
    m_bSerialJoinCount = 0;

    // allow packing of joins by default
    MaxOutputLength = MEDIUM_BLOCK_LEN;

    // get memory for an array of structures for each join type; initialize them to all nulls
    m_pDigital = (DIGITAL_JOIN_STATE *) OsCallocMemory(sizeof(DIGITAL_JOIN_STATE)*m_NumDigitals,1);
    m_pAnalog = (ANALOG_JOIN_STATE *)OsCallocMemory(sizeof(ANALOG_JOIN_STATE)*m_NumAnalogs,1);
    m_pSerial = (SERIAL_JOIN_STATE *)OsCallocMemory(sizeof(SERIAL_JOIN_STATE)*m_NumSerials,1);
    m_pCommands = (COMMAND_RESPONSE *)OsCallocMemory(sizeof(COMMAND_RESPONSE)*m_NumCommands,1);

    if ( !m_pDigital || !m_pAnalog || !m_pSerial || !m_pCommands )
    {
        DmSystemError(DM_ERROR_LEVEL_FATAL, DM_ERROR_SUBSYS_CNET, ERR_CNET_NO_MEMORY, 1);
        return;
    }

    //	clear all joins fields
    memset ((UINT8*)m_pDigital, 0, sizeof(DIGITAL_JOIN_STATE)*m_NumDigitals);
    memset ((UINT8*)m_pAnalog, 0, sizeof(ANALOG_JOIN_STATE)*m_NumAnalogs);
    memset ((UINT8*)m_pSerial, 0, sizeof(SERIAL_JOIN_STATE)*m_NumSerials);
    memset ((UINT8*)m_pCommands, 0, sizeof(COMMAND_RESPONSE)*m_NumCommands);
}
/**
 * \author    Larry Salant
 * \brief     destructor  for class to memorize join states - used for reporting join changes and
 *            for TJI
 * \date      5/8/2008
**/
CresnetJoinState::~CresnetJoinState()
{
#ifndef NO_DESTRUCTOR_NECESSARY
    OsFreeMemory((void*)m_pDigital);
    OsFreeMemory((void*)m_pAnalog);
    OsFreeMemory((void*)m_pSerial);
    OsFreeMemory((void*)m_pCommands);
#endif //NO_DESTRUCTOR_NECESSARY
}
/**
 * \author    Adolfo Velasco
 * \brief
 * \date      05/21/2009
 * \param
 * \param
 * \param
 * \return    void
 * \retval    none
 * \note      Jan 13 2009, R.H: Changed digital joins size to 16
 *            bit.
**/
UINT16 CresnetJoinState::GetMaxDigitalCount( void )
{
    return( m_NumDigitals );
}

/**
 * \author    Adolfo Velasco
 * \brief
 * \date      05/21/2009
 * \param
 * \param
 * \param
 * \return    void
 * \retval    none
 * \note      Jan 13 2009, R.H: Changed analog joins size to 16
 *            bit.
**/
UINT16 CresnetJoinState::GetMaxAnalogCount( void )
{
    return( m_NumAnalogs );
}

/**
 * \author    Adolfo Velasco
 * \brief
 * \date      05/21/2009
 * \param
 * \param
 * \param
 * \return    void
 * \retval    none
**/
UINT16 CresnetJoinState::GetMaxSerialCount( void )
{
    return( m_NumSerials );
}

/**
 * \author    Adolfo Velasco
 * \brief
 * \date      05/21/2009
 * \param
 * \param
 * \param
 * \return    void
 * \retval    none
 * \note      Jan 13 2009, R.H: Changed digital joins size to 16
 *            bit.
**/
UINT16 CresnetJoinState::GetUsedDigitalCount( void )
{
    return( m_bDigitalJoinCount );
}

/**
 * \author    Adolfo Velasco
 * \brief
 * \date      05/21/2009
 * \param
 * \param
 * \param
 * \return    void
 * \retval    none
 * \note      Jan 13 2009, R.H: Changed analog joins size to 16
 *            bit.
**/
UINT16 CresnetJoinState::GetUsedAnalogCount( void )
{
    return( m_bAnalogJoinCount );
}

/**
 * \author    Adolfo Velasco
 * \brief
 * \date      05/21/2009
 * \param
 * \param
 * \param
 * \return    void
 * \retval    none
**/
UINT16 CresnetJoinState::GetUsedSerialCount( void )
{
    return( m_bSerialJoinCount );
}
/**
 * \author    Larry Salant
 * \brief     remember the current state of the digital join and mark that it needs to be
 *            sent to the switch
 * \date      5/5/2008
 * \param     m_Stream : which stream this join is for (not used on this card since only 1 stream)
 * \param     join : which join it is
 * \param     on : current value
 * \return    void
 * \retval    none
**/
void CresnetJoinState::CresnetDigitalMemorize(UINT32 join, BOOL on, UINT8 bSource, UINT8 bForce, /* = false */ UINT8 bDoNotSend) /* = false */
{
    UINT16 index = 0;
    DIGITAL_JOIN_STATE *pDigital = m_pDigital;

#ifdef CRESNETDEBUG
    if ( join >= 4000 )
        DmConsolePrintf( " invalid join = %d stream %d\r", join, m_Stream);
#endif

    // find the join - since joins can be any value (in a 4k range) we use a sparsified table
    // find the join # in the table
    while ( index < m_NumDigitals )
    {
        // if at the last entry, join is not in the table, add it
        if ( !pDigital->JoinDefined )
        {
            pDigital->JoinNumber = join;
            pDigital->JoinDefined = true;

            // update the value
            pDigital->CurrentValue = on;
            // save last sender
            pDigital->Source = bSource;

            // mark it to be sent, if it's set (no need to send 0 initially)
            if ( bForce || on )
            {
                if ( !bDoNotSend )
                {
                    pDigital->Changed = true;
                    pDigital->ChangedRTJI = true;
                }
            }

            //Increment the count
            m_bDigitalJoinCount++;
            break;
        }
        // if this is the join number
        else if ( pDigital->JoinNumber == join )
        {
            //Only report join if the value changed or this is forced
            if ( bForce || ( ((BOOL)pDigital->CurrentValue) != on ) )
            {
                pDigital->CurrentValue = on;
                // save last sender
                pDigital->Source = bSource;
                if ( !bDoNotSend )
                {
                    // mark it to be sent
                    pDigital->Changed = true;
                    pDigital->ChangedRTJI = true;
                }
            }
            break;
        }
        index++;
        pDigital++;
    }
    // make sure we haven't run out of entries in the table
    if ( index >= m_NumDigitals )
    {
        DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_CNET,ERR_CNET_MEMORIZE_TABLE,join);
#ifdef CRESNETDEBUG
        DmConsolePrintf( " overflow (%d) join = %d stream %d\r",index, join, m_Stream);
        pDigital = m_pDigital;
        for ( index = 0; index < m_NumDigitals; index++ )
        {
            DmConsolePrintf( " (%d) join = %d \r",index, pDigital->JoinNumber);
            pDigital++;
        }
#endif
    }

    //RTJI processing moved to timer task to fix stack issues when multiple tasks trying to send RTJI
}

/**
 * \author    Larry Salant
 * \brief     remember the current state of the analog join and mark that it needs to be
 *            sent to the switch
 * \date      5/5/2008
 * \param     m_Stream : which stream this join is for (not used on this card since only 1 stream)
 * \param     join : which join it is
 * \param     value : current value
* \param     bSource
* \param     bForce
* \param     rcbclxtime  : rcb/clx time
* \param     rcbflags : rcb flags
* \param     clxflags : clx flag
*
 * \return    void
 * \retval    none
**/
BOOL dbgtrace1;
void CresnetJoinState::CresnetAnalogMemorize(UINT32 join, UINT16 value, UINT8 bSource, UINT8 bForce, /* = false */ UINT32 rcbclxtime,  /* = 0 */ UINT8 rcbFlags, /* = 0 */ UINT8 pkttype, /* pkttype = 0 */ UINT8 bDoNotSend) /* = false */
{
    UINT16 index = 0;
    ANALOG_JOIN_STATE *pAnalog = m_pAnalog;

    // find the join - since joins can be any value (in a 4k range) we use a sparsified table
    // find the join # in the table
    while ( index < m_NumAnalogs )
    {
        // if not in the table, add it
        if ( !pAnalog->JoinDefined )
        {
            pAnalog->JoinNumber = join;
            pAnalog->JoinDefined = true;

            // update the value
            pAnalog->CurrentValue = value;

#ifdef	RCB_ANALOG_SUPPORT
            if ( pkttype )
            {
                pAnalog->Time  = rcbclxtime;
                pAnalog->Flags = rcbFlags;
                pAnalog->JoinPType = pkttype;
            }
            else
            {
                pAnalog->Time  = 0;
                pAnalog->Flags = 0;
                pAnalog->JoinPType = 0;
            }
#else
            //pAnalog->Time  = 0;
            //pAnalog->Flags = 0;
            pAnalog->JoinPType = 0;
#endif
            // save last sender
            pAnalog->Source = bSource;

            // if value is not 0, mark it to be sent
            // (don't need to send 0 values at startup)
            if ( bForce ||0 != value )
            {
                if ( !bDoNotSend )
                {
                    pAnalog->Changed = true;
                    pAnalog->ChangedRTJI = true;
                }
            }
            // save last sender
            pAnalog->Source = bSource;

            //Increment the count
            m_bAnalogJoinCount++;
            break;
        }
        // if this is the join number
        else if ( pAnalog->JoinNumber == join )
        {
            //Only report join if the value, time changed or this is forced
#ifdef	RCB_ANALOG_SUPPORT
            if ( bForce || (pAnalog->CurrentValue != value) || ((pAnalog->Time != rcbclxtime) && pkttype) )
#else
            if ( bForce || (pAnalog->CurrentValue != value) )
#endif
            {

                if ( bForce || (pAnalog->CurrentValue != value) )
                {
                    //	only report RTJI on value change
                    pAnalog->ChangedRTJI = true;
                }

                // update the value
                pAnalog->CurrentValue = value;
                // save last sender
                pAnalog->Source = bSource;

#ifdef	RCB_ANALOG_SUPPORT
                if ( pkttype )
                {
                    pAnalog->Time  = rcbclxtime;
                    pAnalog->Flags = rcbFlags;
                    pAnalog->JoinPType = pkttype;
                }
                else
                {
                    pAnalog->Time  = 0;
                    pAnalog->Flags = 0;
                    pAnalog->JoinPType = 0;
                }
#else
                //pAnalog->Time  = 0;
                //pAnalog->Flags = 0;
                pAnalog->JoinPType = 0;
#endif

                if ( !bDoNotSend )
                {
                    // mark it to be sent
                    pAnalog->Changed = true;
                    //pAnalog->ChangedRTJI = true;
                }
            }
            break;
        }
        index++;
        pAnalog++;
    }

    // make sure we haven't run out of entries in the table
    if ( index >= m_NumAnalogs )
    {
        DmConsolePrintf("Unable to memorize analog join (%d) = %d\r", join, value);
    }

    //RTJI processing moved to timer task to fix stack issues when multiple tasks trying to send RTJI
}

/**
 * \author    Larry Salant
 * \brief     mark that this join needs to be sent to the switch
 *            due to memory limitations, we don't buffer the string.  When we're ready
 *            to build the packet to send, a callback function will be called to build it
 * \date      5/5/2008
 * \param     m_Stream : which stream this join is for  (not used on this card since only 1 stream)
 * \param     join : which join it is
 * \return    void
 * \retval    none
*/
void CresnetJoinState::CresnetSerialMemorize(UINT32 join, UINT8 bSource)
{
    UINT16 index = 0;
    SERIAL_JOIN_STATE *pSerial = m_pSerial;

    // find the join - since joins can be any value (in a 4k range) we use a sparsified table
    // find the join # in the table
    while ( index < m_NumSerials )
    {
        // if not in the table, add it
        if ( !pSerial->JoinDefined )
        {
            pSerial->JoinNumber = join;
            //	mark it to be defined
            pSerial->JoinDefined = true;

            // mark it to be sent
            pSerial->Changed = true;
            pSerial->ChangedRTJI = true;
            pSerial->Source = bSource;

            //Increment the count
            m_bSerialJoinCount++;
            break;
        }
        // if this is the join number
        else if ( pSerial->JoinNumber == join )
        {
            // mark it to be sent
            pSerial->Changed = true;
            pSerial->ChangedRTJI = true;
            // save last sender
            pSerial->Source = bSource;
            break;
        }
        index++;
        pSerial++;
    }
    // make sure we haven't run out of entries in the table
    if ( index  >= m_NumSerials )
        DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_CNET,ERR_CNET_MEMORIZE_TABLE,2);

    //RTJI processing moved to timer task to fix stack issues when multiple tasks trying to send RTJI
}
/**
 * \author    Larry Salant
 * \brief     mark that we need to respond to this command
 *            due to memory limitations, we don't buffer the string.  When we're ready
 *            to build the packet to send,
 * \date      5/5/2008
 * \param     m_Stream : which stream this command is for  (not used on this card since only 1 stream)
 * \param     pPacket - request packet
 * \return    void
 * \retval    none
*/
void CresnetJoinState::CresnetCommandMemorize(UINT8 *pPacket, UINT8 bSource, UINT8 bSlot)
{
    UINT8 index = 0;
    COMMAND_RESPONSE *pCommands = m_pCommands;

    // find the join - since joins can be any value (in a 4k range) we use a sparsified table
    // find the join # in the table
    while ( index < m_NumCommands )
    {
        // if command is in the list already, and its unique
        if ( pCommands->Command == *(pPacket+2) && pCommands->pPacket == NULL )
        {
            // if the command responses are not unique
            // update its parameters
            pCommands->Type = *(pPacket+3);
            pCommands->Transaction  = *(pPacket+5);
            break;
        }
        // if not in the table, add it
        else if ( pCommands->Command == 0 )
        {
            // if the command responses are unique
            // this is true for the capabilities, for example
            if ( *(pPacket+2) == CAPABILITY_REQUEST )
            {
                // update its parameters
                pCommands->Type = *(pPacket+3);
                pCommands->Transaction  = *(pPacket+5);
                pCommands->pPacket = NULL;
            }
            else if ( *(pPacket+2) == EXTENDED_CAPABILITY_REQUEST )
            {
                // update its parameters
                pCommands->Type = *(pPacket+3);
                pCommands->Transaction  = *(pPacket+5);
                pCommands->pPacket = NULL;
            }
            // if its an downstream device on line packet, save just the header
            else if ( *(pPacket+2) == INTERNAL && *(pPacket+3) == INTERNAL_SUBCMD_DIGITAL )
            {
                // update its parameters
                pCommands->Type = *(pPacket+3);
                pCommands->Transaction  = *(pPacket+5);  // downstream device state
                pCommands->pPacket = NULL;
            }
            else // we must save the whole input buffer
            {
                UINT8 byteCnt;

                //Bugzilla 33933 fix
                //[ID] [CNT] [TYPE] therefore total packet length = [CNT] + 2
                byteCnt = pPacket[TJI_LENGTH_BYTE_OFFSET] + 2;
                // add one more character incase someone adds a null terminator.
                if ( MemMgr && (pCommands->pPacket = MemMgr->GetBlock(byteCnt+1)) != 0 )
                {
                    // copy message to buffer;  This buffer will be released after the command is
                    // processed by the cresnet task.
                    memcpy(pCommands->pPacket, pPacket, byteCnt);
                }
            }

            // mark it to be sent
            pCommands->Source = bSource;
            pCommands->slot   = bSlot;
            // do last since the output routine checks if the command not = 0 to determine
            // if it should send it.
            pCommands->Command = *(pPacket+2);
            break;
        }
        index++;
        pCommands++;
    }
    // make sure we haven't run out of entries in the table
    if ( index  >= m_NumCommands )
        DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_CNET,ERR_CNET_MEMORIZE_TABLE,3);
}

/**
 * \author    Larry Salant
 * \brief     get a buffer to output joins
 * \date      8/14/2009
 * \param     pointer to buffer size
 * \return    pointer to buffer
 * \retval    null if unable to get buffer
**/
UINT8 *CresnetJoinState::GetOutputBuffer( UINT8 *pMaxOutput )
{
    UINT8 *pOutputBuffer = NULL;

    if ( MemMgr )
    {
        // try to get a medium length block
        if ( MaxOutputLength >= MEDIUM_BLOCK_LEN )
        {
            *pMaxOutput = MEDIUM_BLOCK_LEN;
            pOutputBuffer = MemMgr->GetBlock(MEDIUM_BLOCK_LEN);
        }
        // if we couldn't get medium buffer, get small one
        if ( !pOutputBuffer )
        {
            *pMaxOutput = SMALL_BLOCK_LEN;
            pOutputBuffer = MemMgr->GetBlock(SMALL_BLOCK_LEN);
        }
    }
    return pOutputBuffer;
}
/**
 * \author    Larry Salant
 * \brief     override the max output buffer size
 * \date      8/14/2009
 * \param     max buffer size
 * \return    void
 * \retval    none
**/
void CresnetJoinState::SetMaxOutputBuffer( UINT8 MaxOutput )
{
    MaxOutputLength = MaxOutput;
}

/**
* \author    Larry Salant
* \brief     output a digital if any pending
* \date      5/5/2008
* \param     none
* \return    UINT8
* \retval    0 if nothing sent, otherwise 1
**/
UINT8 CresnetJoinState::CresnetNextDigitalPending(void)
{
#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT
    UINT8 bDMNetIdx;
    CDMSwitchCard *pCardPtr     = NULL;
#endif

    UINT8 i;
    UINT8 j;
    UINT32 sTempJoin;
    UINT8* pPacketByte;

    UINT16 bIndex;
    UINT8 bSent = 0;
    DIGITAL_JOIN_STATE *pDigital;

    //Loop variables
    UINT8 bComplete;
    UINT8 bFoundFirstJoin;
    UINT8 bPacketSource;
    UINT8 bPacketSlot;
    UINT8 bJoinCount;
    UINT8 bMaxOutput;

    //Cresnet packet variables
    UINT32 lJoin;
    UINT8* pPacket;
    UINT8* pOutput = NULL;
    UINT8* pLength1 = 0;
    UINT8* pLength2 = 0;

    //Supported digital packets
    //[DEST_ID] [LENGTH1] [ENCAPSULATED] [SLOT] [LENGTH2] [TYPE:CNET_DIGITAL] [JOIN_LSB] [JOIN_MSB]
    //[DEST_ID] [LENGTH1] [TYPE:CNET_DIGITAL] [JOIN_LSB] [JOIN_MSB]

    //We will parse all joins and send them out multiple joins in a single packet
    //We need to send different packets for different sources/slots so keep looping until no joins are found

    //Assume not complete
    bComplete = false;
    while ( !bComplete )
    {
        //Reset the array pointer
        pDigital = m_pDigital;

        //Reset flags
        bFoundFirstJoin = false;   //First join not found yet

        //Reset the join count
        bJoinCount = 0;

        //Reset the [LENGTH2] pointer
        pLength2 = 0;

        //See if there are any joins that need to be sent
        for ( bIndex = 0; bIndex < m_NumDigitals; bIndex++ )
        {
            //Save the join #
            lJoin = pDigital->JoinNumber;

            //No more joins in this stream
            //if (lJoin == 0)
            if ( !pDigital->JoinDefined )
                break;

            //Check for a join change and the source queue not full
            if ( pDigital->Changed && !IsCrestnetSlaveQueueFull(m_Stream, pDigital->Source) )
            {
                //Check if the first join was found
                if ( !bFoundFirstJoin )
                {
                    // if we don't have a buffer, get it
                    if ( pOutput == NULL )
                    {
                        //Initialize packet pointers to the buffer
                        pOutput = GetOutputBuffer( &bMaxOutput );
                        if ( !pOutput )
                            return false;

                        pPacket = pOutput;
                        //[DEST_ID]
                        *pPacket++ = CNET_ID_CTRLSYS;   /* dest ID */

                        //[LENGTH1]
                        pLength1 = pPacket;
                        *pPacket++ = 0;        /* packet length - fill in later */
                    }

                    //The first join dictates the source and slot configuration for the packet

                    //Set the found first join flag
                    bFoundFirstJoin = true;

                    //Save the source
                    bPacketSource = pDigital->Source;

                    //Check the slot
                    if ( IsSubSlottedJoin(lJoin) )
                    {
                        //Need to encapsulate to convey slot and subslot info

                        //Save the slot
                        bPacketSlot = SUBSLOT_FROM_JOIN_NUM(lJoin);



                        //Do slot conversion here?

#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT
	    
			if(GetSlotFromDMNetIdx(&bDMNetIdx, bPacketSlot, CtrlrConfig.bMaxOutputSlotNum, 
				   CtrlrConfig.bMaxInputSlotNum)==0)
			    {
				bPacketSlot = bDMNetIdx - 1;
				lJoin =  SET_SUBSLOT_BITS(JOIN_NUM_FROM_SLOTTED_JOIN(lJoin), 
							  bPacketSlot); 	
			    }
#endif                        
                        

                        //[ENCAPSULATED]
                        *pPacket++ = CRESNET_ENCAPSULATED;

                        //[SLOT]
                        *pPacket++ = bPacketSlot;

                        //[LENGTH2]
                        pLength2 = pPacket;
                        *pPacket++ = 0;  // len - fill in later
                    }
                    else
                    {
                        //Set to top level slot
                        bPacketSlot = TOP_LEVEL_SLOT_0;
                    }

                    //[PACKET_TYPE]
                    *pPacket++ = CNET_DIGITAL;   /* packet type */
                }

                //Process the join

                //Check for the same slot and source
                if ( (bPacketSource == pDigital->Source) && (bPacketSlot == SUBSLOT_FROM_JOIN_NUM(lJoin)) )
                {
                    //Increment the join count
                    bJoinCount++;

                    //Always apply the subslot mask
                    lJoin = (UINT32) JOIN_NUM_FROM_SLOTTED_JOIN(lJoin);

                    //[JOIN_LSB]
                    *pPacket++ = (UINT16)(lJoin & 0x00FF);

                    //[JOIN_MSB]
                    *pPacket = ((UINT16)lJoin >> 8) & 0x007F;
                    if ( !pDigital->CurrentValue )
                    {
                        *pPacket |= 0x80;
                    }
                    pPacket++;

                    //Mark the join as sent
                    pDigital->Changed = false;
                }

                //Fill in missing lengths as needed
                if ( pLength1 )
                    *pLength1 = pPacket-(pLength1+1);
                if ( pLength2 )
                    *pLength2 = pPacket-(pLength2+1);

                //Check if we are out of space
                if ( (m_bPackJoinLimit && (bJoinCount >= m_bPackJoinLimit)) || ((*pLength1+CNET_PKT_HEADER) >= (bMaxOutput-CNET_DIGITAL_SIZE_OFFSET)) )
                {
                    //Bugzilla fix
                    //Double check that the queue is still empty
                    if ( !IsCrestnetSlaveQueueFull(m_Stream, bPacketSource) )
                    {
                        //Send the packet out
                        CresnetSlaveEnqueueOutgoingPacket(m_Stream, pOutput, pPacket-pOutput, bPacketSource);
                    }
                    else
                    {
                        //Start at the end of the packet
                        pPacketByte = pPacket;

                        //Unmark all joins that were assumed sent
                        for ( i = 0; i < bJoinCount; i++ )
                        {
                            //Grab the join number
                            pPacketByte--;      //[JOIN_MSB]
                            pPacketByte--;      //[JOIN_LSB]

                            //Calculate the join
                            sTempJoin = ((UINT32)(pPacketByte[0])) | (((UINT32)(pPacketByte[1] & 0x7F))<<8);

                            //	add slot to sTempJoin
                            sTempJoin = SET_SUBSLOT_BITS(sTempJoin, bPacketSlot);

                            //Search for the join
                            for ( j = 0; j <= bIndex; j++ )
                            {
                                //If matched
                                if ( m_pDigital[j].JoinNumber == sTempJoin )
                                {
                                    //Mark as changed b/c we did not send
                                    m_pDigital[j].Changed = true;
                                    break;
                                }
                            }
                        }
                    }

                    //Indicate that we sent a packet
                    bSent = 1;

                    //Reset the found first join flag
                    bFoundFirstJoin = false;

                    //Restart pointer right after [LENGTH1] byte
                    pPacket = pLength1;
                    pPacket++;

                    //Reset the [LENGTH2] pointer
                    pLength2 = 0;

                    //Reset the join count
                    bJoinCount = 0;
                }
            }
            //Increment to the next digital
            pDigital++;
        }

        //Check if joins were found and need to be sent
        if ( bFoundFirstJoin )
        {
            //Bugzilla fix
            //Double check that the queue is still empty
            if ( !IsCrestnetSlaveQueueFull(m_Stream, bPacketSource) )
            {
#ifdef PROAMP_7X400
                // R.H. Nov 19 2009 - Temporary solution:
                // In case 2 packets of joins send one after the other,
                // the second packet overlaps the first one.
                // Use a delay to let the first packet finish before sending the second.
                HwDelayMsec (100);
#endif // PROAMP_7X400

                //Send the packet out
                CresnetSlaveEnqueueOutgoingPacket(m_Stream, pOutput, pPacket-pOutput, bPacketSource);
            }
            else
            {
                //Start at the end of the packet
                pPacketByte = pPacket;

                //Unmark all joins that were assumed sent
                for ( i = 0; i < bJoinCount; i++ )
                {
                    //Grab the join number
                    pPacketByte--;      //[JOIN_MSB]
                    pPacketByte--;      //[JOIN_LSB]

                    //Calculate the join
                    sTempJoin = ((UINT32)(pPacketByte[0])) | (((UINT32)(pPacketByte[1] & 0x7F))<<8);

                    //	add slot to sTempJoin
                    sTempJoin = SET_SUBSLOT_BITS(sTempJoin, bPacketSlot);

                    //Search for the join
                    for ( j = 0; j <= bIndex; j++ )
                    {
                        //If matched
                        if ( m_pDigital[j].JoinNumber == sTempJoin )
                        {
                            //Mark as changed b/c we did not send
                            m_pDigital[j].Changed = true;
                            break;
                        }
                    }
                }
            }

            //Indicate that we sent a packet
            bSent = 1;

            //Reset the found first join flag
            bFoundFirstJoin = false;

            //Restart pointer right after [LENGTH1] byte
            pPacket = pLength1;
            pPacket++;

            //Reset the [LENGTH2] pointer
            pLength2 = 0;

            //Reset the join count
            bJoinCount = 0;
        }
        //Joins were not found
        else
        {
            //Set the complete flag so we can exit the while loop
            bComplete = true;
        }
    }
    // release the buffer
    if ( pOutput != NULL )
        MemMgr->FreeBlock(pOutput);

    //RTJI is independent of Cresnet so check it separately
    if ( pfRTJIDigitalPending )
        pfRTJIDigitalPending((void*)m_pDigital, m_NumDigitals, m_Stream);

    //Return the sent status
    return bSent;
}

/**
* \author    Larry Salant
* \brief     output an analog, if any pending
* \date      5/6/2008
* \param     none
* \return    UINT8
* \retval    0 if nothing sent, otherwise 1
**/
UINT8 CresnetJoinState::CresnetNextAnalogPending(void)
{
#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT
    UINT8 bDMNetIdx;
    CDMSwitchCard *pCardPtr     = NULL;
#endif
  
    UINT8 j;
    UINT8 i;
    UINT32 sTempJoin;
    UINT8* pPacketByte;

    UINT16 bIndex;
    UINT8 bSent = 0;
    ANALOG_JOIN_STATE *pAnalog;

    //Loop variables
    UINT8 bComplete;
    UINT8 bFoundFirstJoin;
    UINT8 bPacketSource;
    UINT8 bPacketSlot;
    UINT8 bMaxOutput;
    UINT8 bJoinCount;
    UINT8* pOutput = NULL;

    //Cresnet packet variables
    UINT32 lJoin;
    UINT8* pPacket;
    UINT8* pLength1 = 0;
    UINT8* pLength2 = 0;
    UINT32 RCBSharedTime = 0;
    UINT8  reqdatasize;
    UINT32 time = 0;
    UINT8 RCBPktType;

    //Supported analog packets
    //[DEST_ID] [LENGTH1] [ENCAPSULATED] [SLOT] [LENGTH2] [TYPE:ANALOG_SYMMETRIC] [JOIN_MSB] [JOIN_LSB] [VAL_MSB] [VAL_LSB]
    //[DEST_ID] [LENGTH1] [TYPE:ANALOG_SYMMETRIC] [JOIN_MSB] [JOIN_LSB] [VAL_MSB] [VAL_LSB]
    //
    //RCB - [DEST_ID] [LENGTH1] [TYPE:ANALOG_RCB] [TIME3][TIME2][TIME1][TIME0] [JOIN_MSB] [JOIN_LSB] [VAL_MSB] [VAL_LSB] [FLAG]
    //CLX - [DEST_ID] [LENGTH1] [TYPE:ANALOG_CLX] [TIME2][TIME1][TIME0] [JOIN1] [VAL1] [JOIN2] [VAL2] [JOIN3] [VAL3]

    //We will parse all joins and send them out multiple joins in a single packet
    //We need to send different packets for different sources/slots so keep looping until no joins are found

    //Assume not complete
    bComplete = false;
    while ( !bComplete )
    {
        //Reset the array pointer
        pAnalog = m_pAnalog;

        //Reset flags
        bFoundFirstJoin = false;    //First join not found yet

        //Reset the join count
        bJoinCount = 0;

        //Reset the [LENGTH2] pointer
        pLength2 = 0;

        //See if there are any joins that need to be sent
        for ( bIndex = 0; bIndex < m_NumAnalogs; bIndex++ )
        {
            //Save the join #
            lJoin = pAnalog->JoinNumber;

            //No more joins in this stream
            //if (lJoin == 0)
            if ( !pAnalog->JoinDefined )
                break;

            //Check for a join change and the source queue not full
            if ( pAnalog->Changed && !IsCrestnetSlaveQueueFull(m_Stream, pAnalog->Source) )
            {
                //Check if the first join was found
                if ( !bFoundFirstJoin )
                {
                    // start new packet, build packet header

                    // if we don't have a buffer, get it
                    if ( pOutput == NULL )
                    {
                        //Initialize packet pointers to the buffer
                        pOutput = GetOutputBuffer( &bMaxOutput );
                        if ( !pOutput )
                            return false;

                        pPacket = pOutput;
                        //[DEST_ID]
                        *pPacket++ = CNET_ID_CTRLSYS;   /* dest ID */

                        //[LENGTH1]
                        pLength1 = pPacket;
                        *pPacket++ = 0;        /* packet length - fill in later */
                    }

                    //The first join dictates the source and slot configuration for the packet

                    //Set the found first join flag
                    bFoundFirstJoin = true;

                    //Save the source
                    bPacketSource = pAnalog->Source;

                    //Check the slot
                    if ( IsSubSlottedJoin(lJoin) )
                    {
                        //Need to encapsulate to convey slot and subslot info

                        //Save the slot
                        bPacketSlot = SUBSLOT_FROM_JOIN_NUM(lJoin);

                        //Do slot conversion here?

#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT
	    
			if(GetSlotFromDMNetIdx(&bDMNetIdx, bPacketSlot, CtrlrConfig.bMaxOutputSlotNum, 
				   CtrlrConfig.bMaxInputSlotNum)==0)
			    {
				bPacketSlot = bDMNetIdx - 1;
				lJoin =  SET_SUBSLOT_BITS(JOIN_NUM_FROM_SLOTTED_JOIN(lJoin), 
							  bPacketSlot); 	
			    }
#endif                        
                        
                        //[ENCAPSULATED]
                        *pPacket++ = CRESNET_ENCAPSULATED;

                        //[SLOT]
                        *pPacket++ = bPacketSlot;

                        //[LENGTH2]
                        pLength2 = pPacket;
                        *pPacket++ = 0;  // len - fill in later
                    }
                    else
                    {
                        //Set to top level slot
                        bPacketSlot = TOP_LEVEL_SLOT_0;
                    }

#ifdef	RCB_ANALOG_SUPPORT
                    RCBSharedTime = pAnalog->Time;
                    RCBPktType    = pAnalog->JoinPType;
#else
                    RCBSharedTime = 0;
                    RCBPktType    = 0;
#endif

                    if ( RCBPktType )
                    {
                        //	add cmd and time
                        if ( RCBPktType == ANALOG_PKT_TYPE_CLX )
                        {
                            //	add CLX cmd
                            *pPacket++ = CLX_CMD;
                            //	CLX time is only 3-bytes
                        }
                        else
                        {
                            //	add RCB cmd
                            *pPacket++ = RCB_CMD;
                            //	RCB time is 4-bytes	(add MSB byte)
                            *pPacket++ =  HIUINT8(HIUINT16(RCBSharedTime));
                        }
                        //	add time
                        *pPacket++ =  LOUINT8(HIUINT16(RCBSharedTime));
                        *pPacket++ =  HIUINT8(LOUINT16(RCBSharedTime));
                        *pPacket++ =  LOUINT8(LOUINT16(RCBSharedTime));
                    }
                    else
                    {
                        //	send it as standard analog packet
                        //[PACKET_TYPE]
                        *pPacket++ = ANALOG_SYMMETRIC;   /* packet type */
                    }
                }

#ifdef	RCB_ANALOG_SUPPORT
                time = pAnalog->Time;
#else
                time = 0;
#endif
                //Check for the same slot and source
                if ( (bPacketSource == pAnalog->Source) && (bPacketSlot == SUBSLOT_FROM_JOIN_NUM(lJoin)) && (RCBSharedTime == time) && (pAnalog->JoinPType == RCBPktType) )
                {
                    //	save analog join pointer

                    //Increment the join count
                    bJoinCount++;

                    //Always apply the subslot mask
                    lJoin = (UINT32)JOIN_NUM_FROM_SLOTTED_JOIN(lJoin);

                    if ( pAnalog->JoinPType == ANALOG_PKT_TYPE_CLX )
                    {
                        //[JOIN_LSB]
                        *pPacket++ = LOUINT8(lJoin);
                        //[VAL_MSB]
                        *pPacket++ = HIUINT8(pAnalog->CurrentValue);
                    }
                    else
                    {
                        //[JOIN_MSB]
                        *pPacket++ = HIUINT8(lJoin);

                        //[JOIN_LSB]
                        *pPacket++ = LOUINT8(lJoin);

                        //[VAL_MSB]
                        *pPacket++ = HIUINT8(pAnalog->CurrentValue);

                        //[VAL_LSB]
                        *pPacket++ = LOUINT8(pAnalog->CurrentValue);
                    }

                    if ( pAnalog->JoinPType == ANALOG_PKT_TYPE_RCB )
                    {
                        // Add RCB [FLAG]
#ifdef	RCB_ANALOG_SUPPORT
                        *pPacket++ = pAnalog->Flags;
#else
                        *pPacket++ = 0;
#endif

                    }

                    //Mark the join as sent
                    pAnalog->Changed = false;
                }

                //Fill in missing lengths as needed
                if ( pLength1 )
                    *pLength1 = pPacket-(pLength1+1);
                if ( pLength2 )
                    *pLength2 = pPacket-(pLength2+1);

                if ( pAnalog->JoinPType == ANALOG_PKT_TYPE_RCB )
                {
                    reqdatasize = CNET_ANALOG_RCB_SIZE_OFFSET;
                }
                else if ( pAnalog->JoinPType == ANALOG_PKT_TYPE_CLX )
                {
                    reqdatasize = CNET_ANALOG_CLX_SIZE_OFFSET;
                }
                else
                {
                    reqdatasize = CNET_ANALOG_SIZE_OFFSET;
                }

                //Check if we are out of space
                //(lenght adjusted for 2-byte packet header, effected when odd data parameters are added to the packet)
                if ( (m_bPackJoinLimit && (bJoinCount >= m_bPackJoinLimit)) || ((*pLength1+CNET_PKT_HEADER) >= (bMaxOutput-reqdatasize)) )
                {
                    //Bugzilla fix
                    //Double check that the queue is still empty
                    if ( !IsCrestnetSlaveQueueFull(m_Stream, bPacketSource) )
                    {
                        //DmConsolePrintf ("p1 %02x %02x%02x%02x%02x%02x%02x%02x\r\n", bMaxOutput, *pOutput, *(pOutput+1), *(pOutput+2), *(pOutput+3), *(pOutput+4), *(pOutput+5), *(pOutput+6));

                        //Send the packet out
                        CresnetSlaveEnqueueOutgoingPacket(m_Stream, pOutput, pPacket-pOutput, bPacketSource);
                    }
                    else
                    {
                        //	the data could not be posted there is no room in buffer
                        //Start at the end of the packet
                        pPacketByte = pPacket;

                        //Unmark all joins that were assumed sent
                        for ( i = 0; i < bJoinCount; i++ )
                        {
                            //	Get join number
                            if ( pAnalog->JoinPType == ANALOG_PKT_TYPE_CLX )
                            {
                                // CLX packet
                                pPacketByte--;      //[VAL]
                                pPacketByte--;      //[JOIN]

                                //Calculate the join
                                sTempJoin = (UINT32) pPacketByte[0];
                            }
                            else
                            {
                                if ( pAnalog->JoinPType == ANALOG_PKT_TYPE_RCB )
                                {
                                    //	RCB packet
                                    pPacketByte--;  //	Grab flags
                                }
                                else
                                {
                                    // analog packet
                                }

                                //Grab the join number
                                pPacketByte--;      //[VAL_LSB]
                                pPacketByte--;      //[VAL_MSB]
                                pPacketByte--;      //[JOIN_LSB]
                                pPacketByte--;      //[JOIN_MSB]

                                //Calculate the join
                                sTempJoin = (((UINT32)(pPacketByte[0]))<<8) | ((UINT32)(pPacketByte[1]));
                            }

                            //	add slot to sTempJoin
                            sTempJoin = SET_SUBSLOT_BITS(sTempJoin, bPacketSlot);

                            //Search for the join
                            for ( j = 0; j <= bIndex; j++ )
                            {
                                //If matched
                                if ( m_pAnalog[j].JoinNumber == sTempJoin )
                                {
                                    //Mark as changed b/c we did not send
                                    m_pAnalog[j].Changed = true;
                                    break;
                                }
                            }
                        }
                    }

                    //Indicate that we sent a packet
                    bSent = 1;

                    //Reset the found first join flag
                    bFoundFirstJoin = false;

                    //Restart pointer right after [LENGTH1] byte
                    pPacket = pLength1;
                    pPacket++;

                    //Reset the [LENGTH2] pointer
                    pLength2 = 0;

                    //Reset the join count
                    bJoinCount = 0;
                }
            }
            //Increment to the next analog
            pAnalog++;
        }

        //Check if joins were found and need to be sent
        if ( bFoundFirstJoin )
        {
            //Bugzilla fix
            //Double check that the queue is still empty
            if ( !IsCrestnetSlaveQueueFull(m_Stream, bPacketSource) )
            {
                //DmConsolePrintf ("p1 %02x %02x%02x%02x%02x%02x%02x%02x\r\n", bMaxOutput, *pOutput, *(pOutput+1), *(pOutput+2), *(pOutput+3), *(pOutput+4), *(pOutput+5), *(pOutput+6));

                //Send the packet out
                CresnetSlaveEnqueueOutgoingPacket(m_Stream, pOutput, pPacket-pOutput, bPacketSource);
            }
            else
            {
                //	the data could not be posted there is no room in buffer

                //Start at the end of the packet
                pPacketByte = pPacket;

                //Unmark all joins that were assumed sent
                for ( i = 0; i < bJoinCount; i++ )
                {
                    // Get join
                    if ( pAnalog->JoinPType == ANALOG_PKT_TYPE_CLX )
                    {
                        // CLX join
                        pPacketByte--;      //[VAL]
                        pPacketByte--;      //[JOIN]

                        //Calculate the join
                        sTempJoin = (UINT32) pPacketByte[0];
                    }
                    else
                    {
                        if ( pAnalog->JoinPType == ANALOG_PKT_TYPE_RCB )
                        {
                            //	RCB packet
                            pPacketByte--;  //	Grab flags
                        }
                        else
                        {
                            //	Analog packet
                        }

                        //Grab the join number
                        pPacketByte--;      //[VAL_LSB]
                        pPacketByte--;      //[VAL_MSB]
                        pPacketByte--;      //[JOIN_LSB]
                        pPacketByte--;      //[JOIN_MSB]

                        //Calculate the join
                        sTempJoin = (((UINT32)(pPacketByte[0]))<<8) | ((UINT32)(pPacketByte[1]));
                    }

                    //	add slot to sTempJoin
                    sTempJoin = SET_SUBSLOT_BITS(sTempJoin, bPacketSlot);

                    //Search for the join
                    for ( j = 0; j <= bIndex; j++ )
                    {
                        //If matched
                        if ( m_pAnalog[j].JoinNumber == sTempJoin )
                        {
                            //Mark as changed b/c we did not send
                            m_pAnalog[j].Changed = true;
                            break;
                        }
                    }
                }
            }

            //Indicate that we sent a packet
            bSent = 1;

            //Reset the found first join flag
            bFoundFirstJoin = false;

            //Restart pointer right after [LENGTH1] byte
            pPacket = pLength1;
            pPacket++;

            //Reset the [LENGTH2] pointer
            pLength2 = 0;

            //Reset the join count
            bJoinCount = 0;
        }
        //Joins were not found
        else
        {
            //Set the complete flag so we can exit the while loop
            bComplete = true;
        }
    }

    // release the buffer
    if ( pOutput != NULL )
        MemMgr->FreeBlock(pOutput);

    //RTJI is independent of Cresnet so check it separately
    if ( pfRTJIAnalogPending )
        pfRTJIAnalogPending((void*)m_pAnalog, m_NumAnalogs, m_Stream);

    //Return the sent status
    return bSent;
}

/**
* \author    Larry Salant
* \brief     output a serial join, if any pending, in any of the streams
* \date      5/6/2008
* \param     none
* \return    UINT8
* \retval    0 if nothing sent, otherwise 1
**/
UINT8 CresnetJoinState::CresnetNextSerialPending(void)
{
    UINT16 bIndex;
    UINT8 bSent = 0;
    UINT8 *pString;
    UINT16 lLength;
    SERIAL_JOIN_STATE *pSerial = m_pSerial;

    // see if there are any joins that need to be sent
    for ( bIndex = 0; bIndex < m_NumSerials; bIndex++ )
    {
        // if no more joins in this stream
        //if (pSerial->JoinNumber == 0)
        if ( !pSerial->JoinDefined )
            break;
        // if the join has changed and if there is room in the queue for this stream
        else if ( pSerial->Changed && !IsCrestnetSlaveQueueFull(m_Stream, pSerial->Source) )
        {
            // get the string
            pString = GetCresnetSerialString(m_Stream, pSerial->JoinNumber, &lLength);

            //Allow TJI null strings (join # w/o any serial data)
            if ( pString )
            {
                // send join
                if ( 0 == CresnetSlaveSendSerial(m_Stream, pString, lLength, pSerial->JoinNumber, pSerial->Source) )
                {
                    // indicate that we sent a packet
                    bSent = 1;
                    // indicate that it has been processed
                    pSerial->Changed = false;
                }
            }
            else
            {
                // indicate that it has been processed
                pSerial->Changed = false;
            }
        }

        //Check for an RTJI change
        if ( pSerial->ChangedRTJI )
        {
            // if RTJI supported, process packet if stream being monitored.
            if ( pfRTJIMonitor )
                (*pfRTJIMonitor)(TJI_SUBTYPE_SERIAL_OUTPUT, m_Stream, pSerial->JoinNumber, 0);

            //Clear the flag
            pSerial->ChangedRTJI = false;
        }

        //Increment to the next serial
        pSerial++;
    }
    return bSent;
}
/**
 * \author    Larry Salant
 * \brief     output the response to a command, if any pending in any stream
 * \date      5/7/2008
 * \param     none
 * \return    UINT8
 * \retval    0 if nothing sent, otherwise 1
**/
UINT8 CresnetJoinState::CresnetNextCommandPending(void)
{
    COMMAND_RESPONSE *pCommands = m_pCommands;
    UINT8 bIndex;
    UINT8 bSent = 0;

    // see if there are any joins that need to be sent
    for ( bIndex = 0; bIndex < m_NumCommands; bIndex++ )
    {
        // if there is a command to send
        if ( pCommands->Command != 0 )
        {
            // if there is room in the queue for this stream
            if ( !IsCrestnetSlaveQueueFull(m_Stream, pCommands->Source) )
            {
                // if the commands is Capabilities
                if ( pCommands->Command == CAPABILITY_REQUEST )
                {
                    if ( 0 == CresnetSlaveSendCapabilities(m_Stream, pCommands) )
                    {
                        if( pDMNetMapper )
                        {
                            // resend the status of the downstream device to the controller, if necessary
                            pDMNetMapper->ReportOnlineStatus();
                        }

                        // remove command from the list
                        pCommands->Command = 0;

                        // indicate that we sent a packet
                        bSent = 1;
                    }
                }
                else if ( pCommands->Command == EXTENDED_CAPABILITY_REQUEST )
                {
                    if ( 0 == CresnetSlaveSendExtendedCapabilities(m_Stream, pCommands) )
                    {
                        // remove command from the list
                        pCommands->Command = 0;

                        // indicate that we sent a packet
                        bSent = 1;
                    }
                }
                // if the commands is Online
                else if ( pCommands->Command == INTERNAL && pCommands->Type == INTERNAL_SUBCMD_DIGITAL )
                {
                    // if device on line
                    if ( pCommands->Transaction )
                        // send Downstream Device is Online
                        CresnetSlaveEnqueueOutgoingPacket(m_Stream, (void*)EncapOnlinePacket,sizeof(EncapOnlinePacket), pCommands->Source);
                    else // send Downstream Device is Offline
                        CresnetSlaveEnqueueOutgoingPacket(m_Stream, (void*)EncapOfflinePacket,sizeof(EncapOnlinePacket), pCommands->Source);

                    // remove command from the list
                    pCommands->Command = 0;

                    // indicate that we sent a packet
                    bSent = 1;
                }
                else // it must be a TJI command
                {
                    // process the command
                    ProcessRequestTJI( pCommands->pPacket, pCommands->Source, pCommands->slot );

                    // release the buffer
                    if ( MemMgr )
                        MemMgr->FreeBlock(pCommands->pPacket);

                    // remove command from the list
                    pCommands->Command = 0;

                    // indicate that we sent a packet
                    bSent = 1;
                }
            }
        }
        //else no room, wait for next time

        // see if there are more commands
        pCommands++;
    }
    return bSent;
}
/**
 * \author    Larry Salant
 * \brief     gets the value of the specified digitial join
 * \date      5/13/2008
 * \param     join : which join it is
 * \return    UINT8
 * \retval    1 or 0 - join value; if join not found, returns 0
**/
UINT8 CresnetJoinState::dig_val_in(UINT32 sJoin, UINT8* pStatus)
{
    UINT16 bIndex;
    DIGITAL_JOIN_STATE *pDigital = m_pDigital;

    //Verify parameters
    if ( !pStatus )
        return 0;

    // see if there are any joins that need to be sent
    for ( bIndex = 0; bIndex < m_NumDigitals; bIndex++ )
    {
        // if this is the join,
        if ( pDigital->JoinNumber == sJoin )
        {
            //Return the valid join
            *pStatus = true;
            return pDigital->CurrentValue;
        }
        // if there are no more joins in the list
        //else if (pDigital->JoinNumber == 0)
        else if ( !pDigital->JoinDefined )
            break; // quit
        pDigital++;
    }

    //Return invalid join
    *pStatus = false;
    return 0;
}

/**
 * \author    Larry Salant
 * \brief     gets the value of the specified analog join
 * \date      5/13/2008
 * \param     join : which join it is
 * \return    UINT16
 * \retval    join value; if join not found, returns 0
**/
UINT16 CresnetJoinState::get_analog_in_value(UINT32 sJoin, UINT8* pStatus)
{
    UINT16 bIndex;
    ANALOG_JOIN_STATE *pAnalog = m_pAnalog;

    //Verify parameters
    if ( !pStatus )
        return 0;

    // see if there are any joins that need to be sent
    for ( bIndex = 0; bIndex < m_NumAnalogs; bIndex++ )
    {
        // if this is the join,
        if ( pAnalog->JoinNumber == sJoin )
        {
            //Return the valid join
            *pStatus = true;
            return pAnalog->CurrentValue;
        }
        // if there are no more joins in the list
        //else if (pAnalog->JoinNumber == 0)
        else if ( !pAnalog->JoinDefined )
            break; // quit
        pAnalog++;
    }

    //Return invalid join
    *pStatus = false;
    return 0;
}

/**
* \author     Adolfo Velasco
* \brief      Process the TJI packet request and prepares the appropriate TJI response
* \date       9/18/06
* \param      packet - pointer to the packet
* \param      bCresnet - source of request - 0 = console, otherwise destination (Cresnet, CIP)
* \param      slot - subslot that the joins belong to
* \return     void
* \retval     none
*/
void CresnetJoinState::ProcessRequestTJI( unsigned char* packet, UINT8 bCresnet,  UINT8 bSlot )
{
    //unsigned long source = *packet;
    int   packlen = packet[1] + 2;           // temp variables for easier debug
    int   datalen = packlen - TJI_PACKET_HEADER_SIZE;
    unsigned char subType = packet[3];
    unsigned char flags = packet[4];  // two bits TJF_FLAG_PRESERVE_BITS must be returnd
    unsigned char transID = packet[5];
    UINT16 dataCount;
    UINT32 join;

    //Response packet variables
    unsigned char rFlag = (packet[4] & TJF_FLAG_PRESERVE_BITS);  //Response flag
    unsigned char *rData;
    UINT16 i, j = 0;
    UINT16 sMin, sMax;
    UINT8 *pText;
    UINT8 bStatus;
    UINT8 bMore;
    UINT8 bDigitalState;
    UINT16 sAnalogState;
    UINT8 bInvalidSubSlot;

    //First check for errors that apply to digital, analog and serial requests
    if ( subType <= TJI_SUBTYPE_SERIAL_OUTPUT )
    {
        //First check that data length is valid
        if ( datalen % 2 == 1 )     //Data is populated with 16-bit join #'s
        {
            rFlag |= TJI_FLAG_BIT_ERROR;
        }

        //Range
        else if ( flags & TJI_FLAG_BIT_MOD )
        {
            //Get the min and max of the range
            sMin = LE2NATIVE16(*(unsigned short*) &packet[6]);
            sMax = LE2NATIVE16(*(unsigned short*) &packet[8]);

            //Check range
            if ( sMax < sMin || datalen != 4 )
            {
                rFlag |= TJI_FLAG_BIT_JRV;
                rFlag |= TJI_FLAG_BIT_ERROR;
            }
        }
    }

    //If no error in packet, continue processing the request
    if ( !(rFlag & TJI_FLAG_BIT_ERROR) )
    {
        // get a temporary buffer
        if ( MemMgr && (rData = MemMgr->GetBlock(TJI_PACKET_MAX_TOTAL_SIZE)) != 0 )
        {
            // process based on subtype
            //DM Cards do not differentiate between input/output join values
            switch ( subType )
            {
                case TJI_SUBTYPE_DIGITAL_INPUT:
                case TJI_SUBTYPE_DIGITAL_OUTPUT:
                    {
                        //Range
                        if ( flags & TJI_FLAG_BIT_MOD )
                        {
                            //sMin and sMax have been checked and are valid
                            for ( i = 0; i <= (sMax - sMin); i++ )
                            {
                                join = sMin+i;

                                //Get the LSB/MSB of the join
                                *(unsigned short*) &rData[j] = LE2NATIVE16(join);

                                //Assume not invalid subslot
                                bInvalidSubSlot = false;
                                if ( bSlot != TOP_LEVEL_SLOT_0 )//if sub-slot
                                {
                                    join = ApplySubSlotMaskToJoinNumber( bSlot, join );

                                    //Check for errors
                                    if ( !IsSubSlottedJoin(join) )
                                        bInvalidSubSlot = true;
                                }

                                //Get the join state
                                bDigitalState = dig_val_in(join, &bStatus);

                                //If join is low, set the MSB
                                if ( !bDigitalState )
                                    rData[j+1] |= 0x80;

                                //Increment next word
                                j+=2;

                                //Verify the status
                                if ( !bStatus || bInvalidSubSlot )
                                {
                                    //Check if good data exists
                                    if ( j > DM_TJI_DIGITAL_JOIN_BYTES )
                                    {
                                        //Send it immediately with the flag cleared
                                        rFlag &= ~(TJI_FLAG_BIT_JRV);
                                        ProcessResponseTJI( subType, rFlag, transID, &rData[0], NULL, j-DM_TJI_DIGITAL_JOIN_BYTES, true, bCresnet, m_Stream );
                                    }

                                    //Send the join with the error
                                    rFlag |= TJI_FLAG_BIT_JRV;
                                    bMore = ( (i < (sMax - sMin)) ? true : false );
                                    ProcessResponseTJI( subType, rFlag, transID, &rData[j-DM_TJI_DIGITAL_JOIN_BYTES], NULL, DM_TJI_DIGITAL_JOIN_BYTES, bMore, bCresnet, m_Stream );

                                    //Reset the data and flags
                                    j = 0;
                                    rFlag &= ~(TJI_FLAG_BIT_JRV);
                                }

                                //Check if data limit has been reached
                                if ( j > (TJI_PACKET_DATA_MAX_SIZE-DM_TJI_DIGITAL_JOIN_BYTES) )
                                {
                                    ProcessResponseTJI( subType, rFlag, transID, &rData[0], NULL, j, true, bCresnet, m_Stream );
                                    j = 0;
                                }
                            }
                        }

                        //List
                        else
                        {
                            for ( i = TJI_PACKET_HEADER_SIZE; i < packlen; i+=2 )
                            {
                                //Get the LSB/MSB of the join
                                *(unsigned short*) &rData[j] = *(unsigned short*) &packet[i];
                                // We should mask after converting and not before !!!!!
                                join = LE2NATIVE16((*(unsigned short*) &packet[i])) & 0x7FFF;

                                //Assume not invalid subslot
                                bInvalidSubSlot = false;
                                if ( bSlot != TOP_LEVEL_SLOT_0 )//if sub-slot
                                {
                                    join = ApplySubSlotMaskToJoinNumber( bSlot, join );

                                    //Check for errors
                                    if ( !IsSubSlottedJoin(join) )
                                        bInvalidSubSlot = true;
                                }

                                //Get the join state
                                bDigitalState = dig_val_in(join, &bStatus);

                                //If join is low, set the MSB
                                if ( !bDigitalState )
                                    rData[j+1] |= 0x80;

                                //Increment next word
                                j+=2;

                                //Verify the status
                                if ( !bStatus || bInvalidSubSlot )
                                {
                                    //Check if good data exists
                                    if ( j > DM_TJI_DIGITAL_JOIN_BYTES )
                                    {
                                        //Send it immediately with the flag cleared
                                        rFlag &= ~(TJI_FLAG_BIT_JRV);
                                        ProcessResponseTJI( subType, rFlag, transID, &rData[0], NULL, j-DM_TJI_DIGITAL_JOIN_BYTES, true, bCresnet, m_Stream );
                                    }

                                    //Send the join with the error
                                    rFlag |= TJI_FLAG_BIT_JRV;
                                    bMore = ( (i < (packlen - 2)) ? true : false );
                                    ProcessResponseTJI( subType, rFlag, transID, &rData[j-DM_TJI_DIGITAL_JOIN_BYTES], NULL, DM_TJI_DIGITAL_JOIN_BYTES, bMore, bCresnet, m_Stream );

                                    //Reset the data and flags
                                    j = 0;
                                    rFlag &= ~(TJI_FLAG_BIT_JRV);
                                }

                                //Check if data limit has been reached
                                if ( j > (TJI_PACKET_DATA_MAX_SIZE-DM_TJI_DIGITAL_JOIN_BYTES) )
                                {
                                    ProcessResponseTJI( subType, rFlag, transID, &rData[0], NULL, j, true, bCresnet, m_Stream );
                                    j = 0;
                                }
                            }
                        }

                        //Send the last packet
                        if ( j > 0 )
                        {
                            ProcessResponseTJI( subType, rFlag, transID, &rData[0], NULL, j, false, bCresnet, m_Stream );
                        }
                        break;
                    }

                case TJI_SUBTYPE_ANALOG_INPUT:
                case TJI_SUBTYPE_ANALOG_OUTPUT:
                    {
                        //Range
                        if ( flags & TJI_FLAG_BIT_MOD )
                        {
                            //sMin and sMax have been checked and are valid
                            for ( i = 0; i <= (sMax - sMin); i++ )
                            {
                                join = (sMin & 0x7FFF) + i;
                                //Get the LSB/MSB of the join
                                *(unsigned short*) &rData[j] = LE2NATIVE16( join );
                                j+=2;

                                //Assume not invalid subslot
                                bInvalidSubSlot = false;
                                if ( bSlot != TOP_LEVEL_SLOT_0 )//if sub-slot
                                {
                                    join = ApplySubSlotMaskToJoinNumber( bSlot, join );

                                    //Check for errors
                                    if ( !IsSubSlottedJoin(join) )
                                        bInvalidSubSlot = true;
                                }

                                //Get the join state
                                sAnalogState = LE2NATIVE16( get_analog_in_value(join, &bStatus) );

                                //Get the analog value
                                *(unsigned short*) &rData[j] = sAnalogState;
                                j+=2;

                                //Verify the status
                                if ( !bStatus || bInvalidSubSlot )
                                {
                                    //Check if good data exists
                                    if ( j > DM_TJI_ANALOG_JOIN_BYTES )
                                    {
                                        //Send it immediately with the flag cleared
                                        rFlag &= ~(TJI_FLAG_BIT_JRV);
                                        ProcessResponseTJI( subType, rFlag, transID, &rData[0], NULL, j-DM_TJI_ANALOG_JOIN_BYTES, true, bCresnet, m_Stream );
                                    }

                                    //Send the join with the error
                                    rFlag |= TJI_FLAG_BIT_JRV;
                                    bMore = ( (i < (sMax - sMin)) ? true : false );
                                    ProcessResponseTJI( subType, rFlag, transID, &rData[j-DM_TJI_ANALOG_JOIN_BYTES], NULL, DM_TJI_ANALOG_JOIN_BYTES, bMore, bCresnet, m_Stream );

                                    //Reset the data and flags
                                    j = 0;
                                    rFlag &= ~(TJI_FLAG_BIT_JRV);
                                }

                                //Check if data limit has been reached
                                if ( j > (TJI_PACKET_DATA_MAX_SIZE-DM_TJI_ANALOG_JOIN_BYTES) )
                                {
                                    ProcessResponseTJI( subType, rFlag, transID, &rData[0], NULL, j, true, bCresnet, m_Stream );
                                    j = 0;
                                }
                            }
                        }

                        //List
                        else
                        {
                            for ( i = TJI_PACKET_HEADER_SIZE; i < packlen; i+=2 )
                            {
                                //Get the LSB/MSB of the join
                                *(unsigned short*) &rData[j] = *(unsigned short*) &packet[i];
                                j+=2;

                                join = LE2NATIVE16(*(unsigned short*) &packet[i]) & 0xFFFF;

                                //Assume not invalid subslot
                                bInvalidSubSlot = false;
                                if ( bSlot != TOP_LEVEL_SLOT_0 )//if sub-slot
                                {
                                    join = ApplySubSlotMaskToJoinNumber( bSlot, join );

                                    //Check for errors
                                    if ( !IsSubSlottedJoin(join) )
                                        bInvalidSubSlot = true;
                                }

                                //Get the join state
                                sAnalogState = LE2NATIVE16( get_analog_in_value(join, &bStatus) );

                                //Get the analog value
                                *(unsigned short*) &rData[j] = sAnalogState;
                                j+=2;

                                //Verify the status
                                if ( !bStatus || bInvalidSubSlot )
                                {
                                    //Check if good data exists
                                    if ( j > DM_TJI_ANALOG_JOIN_BYTES )
                                    {
                                        //Send it immediately with the flag cleared
                                        rFlag &= ~(TJI_FLAG_BIT_JRV);
                                        ProcessResponseTJI( subType, rFlag, transID, &rData[0], NULL, j-DM_TJI_ANALOG_JOIN_BYTES, true, bCresnet, m_Stream );
                                    }

                                    //Send the join with the error
                                    rFlag |= TJI_FLAG_BIT_JRV;
                                    bMore = ( (i < (packlen - 2)) ? true : false );
                                    ProcessResponseTJI( subType, rFlag, transID, &rData[j-DM_TJI_ANALOG_JOIN_BYTES], NULL, DM_TJI_ANALOG_JOIN_BYTES, bMore, bCresnet, m_Stream );

                                    //Reset the data and flags
                                    j = 0;
                                    rFlag &= ~(TJI_FLAG_BIT_JRV);
                                }

                                //Check if data limit has been reached
                                if ( j > (TJI_PACKET_DATA_MAX_SIZE-DM_TJI_ANALOG_JOIN_BYTES) )
                                {
                                    ProcessResponseTJI( subType, rFlag, transID, &rData[0], NULL, j, true, bCresnet, m_Stream );
                                    j = 0;
                                }
                            }
                        }

                        //Send the last packet
                        if ( j > 0 )
                        {
                            ProcessResponseTJI( subType, rFlag, transID, &rData[0], NULL, j, false, bCresnet, m_Stream );
                        }
                        break;
                    }

                case TJI_SUBTYPE_SERIAL_INPUT:
                case TJI_SUBTYPE_SERIAL_OUTPUT:
                    {
                        //Range
                        if ( flags & TJI_FLAG_BIT_MOD )
                        {
                            //sMin and sMax have been checked and are valid
                            for ( i = 0; i <= (sMax - sMin); i++ )
                            {
                                join = sMin+i;
                                //Send 1 serial join per TJI packet
                                //To save copying the string into a temporary buffer, put the LSB/MSB of the join number
                                // into the header buffer
                                *(unsigned short*) &rData[j] = LE2NATIVE16( join );

                                //Assume not invalid subslot
                                bInvalidSubSlot = false;
                                if ( bSlot != TOP_LEVEL_SLOT_0 )//if sub-slot
                                {
                                    join = ApplySubSlotMaskToJoinNumber( bSlot, join );

                                    //Check for errors
                                    if ( !IsSubSlottedJoin(join) )
                                        bInvalidSubSlot = true;
                                }
                                //Get the string join value, calculate the size, transfer to the data array (will return NULL
                                // if join not found)
                                pText = GetCresnetSerialString(m_Stream, join, &dataCount);

                                //Bugzilla fix
                                //If the join is invalid, pText will be NULL
                                if ( pText && !bInvalidSubSlot )
                                {
                                    //Make sure the error flag is cleared
                                    rFlag &= ~(TJI_FLAG_BIT_JRV);
                                }
                                else
                                {
                                    //Set the error flag
                                    rFlag |= TJI_FLAG_BIT_JRV;
                                }

                                // pass the header and pointer to string to be sent out. We do not need to convert the
                                // string data;  add 2 to dataCount for the join number

                                //Allow TJI null strings (join # w/o any serial data)
                                bMore = ( (i < (sMax - sMin)) ? true : false );
                                ProcessResponseTJI( subType, rFlag, transID, pText, &rData[0], dataCount,
                                                    bMore, bCresnet, m_Stream );    //Last packet, clear the bMore bit
                            }
                        }

                        //List
                        else
                        {
                            for ( i = TJI_PACKET_HEADER_SIZE; i < packlen; i+=2 )
                            {
                                //Get the LSB/MSB of the join
                                *(UINT16*) &rData[j] = *(UINT16*) &packet[i];

                                join = LE2NATIVE16(*(UINT16*)&packet[i]);

                                //Assume not invalid subslot
                                bInvalidSubSlot = false;
                                if ( bSlot != TOP_LEVEL_SLOT_0 )//if sub-slot
                                {
                                    join = ApplySubSlotMaskToJoinNumber( bSlot, join );

                                    //Check for errors
                                    if ( !IsSubSlottedJoin(join) )
                                        bInvalidSubSlot = true;
                                }
                                //Get the string join value, calculate the size, transfer to the data array
                                pText = GetCresnetSerialString(m_Stream, join, &dataCount);

                                //Bugzilla fix
                                //If the join is invalid, pText will be NULL
                                if ( pText && !bInvalidSubSlot )
                                {
                                    //Make sure the error flag is cleared
                                    rFlag &= ~(TJI_FLAG_BIT_JRV);
                                }
                                else
                                {
                                    //Set the error flag
                                    rFlag |= TJI_FLAG_BIT_JRV;
                                }

                                // pass the header and pointer to string to be sent out. We do not need to convert the
                                // string data;  add 2 to dataCount for the join number

                                //Allow TJI null strings (join # w/o any serial data)
                                bMore = ( (i < (packlen - 2)) ? true : false );
                                ProcessResponseTJI( subType, rFlag, transID, pText, &rData[0], dataCount,
                                                    bMore, bCresnet, m_Stream );   //Last packet, clear the bMore bit
                            }
                        }
                        break;
                    }

                case TJI_SUBTYPE_GENERIC_CONFIG:
                    {
                        break;
                    }

                case TJI_SUBTYPE_DEVICE_CONSOLE:
                    {
#ifdef RCON_IMPLEMENTED
                        //Store the transaction ID and flag
                        g_TransTJI = transID;
                        g_FlagTJI = rFlag;

                        CMD_STRUCT RCON_Command;
                        RCON_Command.Source = FROM_TJI_REDIRECT;
                        RCON_Command.Length = packlen - TJI_PACKET_HEADER_SIZE;
                        memcpy(RCON_Command.CommandString, (char*)&packet[TJI_PACKET_HEADER_SIZE], RCON_Command.Length);

                        if ( !TJI_InputCaptured )
                        {
                            RCON_Command.CommandString[RCON_Command.Length] = 0x00;  //NULL terminate the string
                            SendCommandToPipe(&RCON_Command);
                        }
                        else
                        {
                            RCON_Command.CommandString[RCON_Command.Length] = '\r';  //add /r to end line
                            RCON_Command.CommandString[++RCON_Command.Length] = 0x00;  //NULL terminate the string

                            short status = NU_Send_To_Pipe(&CapturedRxPipe,RCON_Command.CommandString,RCON_Command.Length,NU_NO_SUSPEND);
                            if ( status != NU_SUCCESS )
                            {
                                error_printf("rxparse: Error writing to captured Rx pipe. Status=%d ,Char Lost: %u",status,packlen - 4);
                                break;
                            }
                        }
#endif // RCON_IMPLEMENTED
                        break;
                    }
                default:
                    {
                        rFlag |= TJI_FLAG_BIT_UNSUP;

                        //Send the last packet
                        ProcessResponseTJI( subType, rFlag, transID, rData, NULL, j, false, bCresnet, m_Stream );
                        break;
                    }
            }
            // release the buffer
            if ( MemMgr )
                MemMgr->FreeBlock(rData);

        }
    }
}

/**
 * \author    Andrew Salmon
 * \brief     gets the value of the specified digital join
 * \date      09/30/2008
 * \param     join : which join it is
 * \return    BOOL
 * \retval    join value; if join not found, returns 0
**/
BOOL CresnetJoinState::CresnetDigitalRecall( UINT32 sJoin )
{
    UINT8 bStatus;
    return( dig_val_in((UINT32) sJoin, &bStatus) );
}

/**
 * \author    Andrew Salmon
 * \brief     gets the value of the specified analog join
 * \date      12/24/2008
 * \param     join : which join it is
 * \return    UINT16
 * \retval    join value; if join not found, returns 0
**/
UINT16 CresnetJoinState::CresnetAnalogRecall( UINT32 sJoin )
{
    UINT8 bStatus;
    return( get_analog_in_value((UINT32) sJoin, &bStatus) );
}

/**
 * \author    Adolfo Velasco, Roni Hudes
 * \brief     Processes the all clear packet
 *            Sets analog/digital join updates b/c new optimizations only force joins if needed
 * \date      07/13/2009
 * \param     bForce - force to send every single join without exception.
 * \param     bReportSerial - if set also, report serials
 * \return
 * \retval
**/
void CresnetJoinState::ProcessAllClear (UINT8 bForce, /* = false */ UINT8 bReportSerial) /* = false */
{
    UINT16 i;
    DIGITAL_JOIN_STATE* pDigital = m_pDigital;
    ANALOG_JOIN_STATE* pAnalog = m_pAnalog;
    SERIAL_JOIN_STATE*  pSerial = m_pSerial;

    //Check all digital joins
    for ( i = 0; i < m_NumDigitals; i++ )
    {
        //if ((bForce && pDigital->JoinNumber) || // Send every single valid join without exception.
        //    (!bForce && pDigital->JoinNumber && pDigital->CurrentValue && !pDigital->Changed)) // Check for a valid join that's not being reported.
        if ( (bForce && pDigital->JoinDefined) || // Send every single valid join without exception.
             (!bForce && pDigital->JoinDefined && pDigital->CurrentValue && !pDigital->Changed) ) // Check for a valid join that's not being reported.
        {
            //Force it to be reported
            pDigital->Changed = true;
        }

        //Increment to the next join
        pDigital++;
    }

    //Check all analog joins
    for ( i = 0; i < m_NumAnalogs; i++ )
    {
        //if ((bForce && pAnalog->JoinNumber) || // Send every single valid join without exception.
        //    (!bForce && pAnalog->JoinNumber && pAnalog->CurrentValue && !pAnalog->Changed)) //Check for a valid join that's not being reported
        if ( (bForce && pAnalog->JoinDefined) || // Send every single valid join without exception.
             (!bForce && pAnalog->JoinDefined && pAnalog->CurrentValue && !pAnalog->Changed) ) //Check for a valid join that's not being reported
        {
            //Force it to be reported
            pAnalog->Changed = true;
        }

        //Increment to the next join
        pAnalog++;
    }

    //	Check all serial joins
    if ( bReportSerial )
    {
        for ( i = 0; i < m_NumSerials; i++ )
        {
            if ( (bForce && pSerial->JoinDefined) || // Send every single valid join without exception.
                 (!bForce && pSerial->JoinDefined && !pSerial->Changed) ) // Check for a valid join that's not being reported.
            {
                //	Force it to be reported
                pSerial->Changed = true;
            }

            //	Increment to the next join
            pSerial++;
        }
    }
}

/**
 * \author    Adolfo Velasco
 * \brief     Starts the command acknowledge request timer
 *            Default to 500ms per V24
 * \date      08/15/2012
 * \param
 * \return
 * \retval
**/
void CresnetJoinState::StartCommandRequestAckTimer( UINT8 bSource )
{
    //Use 500ms from V24
    m_iCommandRequestAckTimer = (500/DM_CRESNET_SLAVE_SEND_PENDING_TIMER_MS);
    m_bCommandRequestAckSource = bSource;
}

/**
 * \author    Adolfo Velasco
 * \brief     Starts the command acknowledge request timer
 *            Default to 500ms per V24
 * \date      08/15/2012
 * \param
 * \return
 * \retval
**/
void CresnetJoinState::CheckCommandRequestAckTimer( void )
{
    //Check if the timer is running
    if ( m_iCommandRequestAckTimer > 0 )
    {
        //Decrement the timer
        m_iCommandRequestAckTimer--;
    }
    //Check if the timer just completed
    else if ( m_iCommandRequestAckTimer == 0 )
    {
        //Make sure the queue has room
        if ( !IsCrestnetSlaveQueueFull(m_Stream, m_bCommandRequestAckSource) )
        {
            //Disable the timer
            m_iCommandRequestAckTimer = DM_LIBRARY_DEBOUNCE_DISABLED;

            UINT8 p[5];
            p[0] = CNET_ID_CTRLSYS;
            p[1] = 2;
            p[2] = CNET_COMMAND;
            p[3] =  CNET_CMD_REQUEST_END_ACK;

            //Send the packet
            CresnetSlaveEnqueueOutgoingPacket(m_Stream, p, 4, m_bCommandRequestAckSource);
        }
    }
}



